<?php

namespace App\Http\Controllers\Backend\Auth\User;

use App\Models\Auth\{User, UserDetail};
use App\Http\Controllers\Controller;
use App\Events\Backend\Auth\User\UserDeleted;
use App\Repositories\Backend\Auth\RoleRepository;
use App\Repositories\Backend\Auth\UserRepository;
use App\Repositories\Backend\Auth\PermissionRepository;
use App\Http\Requests\Backend\Auth\User\StoreUserRequest;
use App\Http\Requests\Backend\Auth\User\ManageUserRequest;
use App\Http\Requests\Backend\Auth\User\UpdateUserRequest;
use App\Http\Controllers\Backend\helper\UserFunction;
use Illuminate\Support\Facades\Gate;

/**
 * Class UserController.
 */
class UserController extends Controller
{
    /**
     * @var UserRepository
     */
    protected $userRepository;

    /**
     * @var UserFunction
     */
    protected $userFunction;

    /**
     * UserController constructor.
     *
     * @param UserRepository $userRepository
     */
    public function __construct(UserRepository $userRepository, UserFunction $userFunction)
    {
        
        $this->userRepository = $userRepository;
        $this->userFunction = $userFunction;
        //$this->middleware(['role:administrator','permission:publish articles|edit articles'], ['only' => ['adminindex','show']]);
        $this->middleware(['role:superAdmin'], ['only' => ['adminindex']]);
        $this->middleware(['role:superAdmin|admin'], ['only' => ['superstockezindex']]);
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(ManageUserRequest $request)
    {
        
        return view('backend.auth.user.index')
            ->withUsers($this->userRepository->getActivePaginated(25, 'id', 'asc','superAdmin'));
    }

    /**
     * @param ManageUserRequest    $request
     * @param RoleRepository       $roleRepository
     * @param PermissionRepository $permissionRepository
     *
     * @return mixed
     */
    public function create(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository,$userrole=null)
    {
        $sub_selectedUser = $selectedUser ='';
        $parentRole = $this->userFunction->getParentRole($userrole);
        $paricularParentRoleId = ((count(auth()->user()->roles)>0)&&(auth()->user()->roles[0]->id!=1))?auth()->user()->id:0;
        return view('backend.auth.user.create')
            ->with(compact('selectedUser','userrole'))
            ->withUsers($this->userRepository->getParentUser($paricularParentRoleId, 'id', 'asc',$parentRole))
            ->withRoles($roleRepository->with('permissions')->where('name',$userrole)->get(['id', 'name']))
            ->withPermissions($permissionRepository->get(['id', 'name']));
    }

    /**
     * @param StoreUserRequest $request
     *
     * @throws \Throwable
     * @return mixed
     */
    public function store(StoreUserRequest $request)
    {
        
        $roleType = $request->input('roles')[0];
        $this->userRepository->create($request->only(
            'first_name',
            'last_name',
            'email',
            'password',
            'active',
            'confirmed',
            'confirmation_email',
            'roles',
            'permissions',
            'adminParent',
            'subchild'
        ));
        $indexRouter = $this->userFunction->getIndexRouter($roleType);//'admin.auth.user.admin';
        return redirect()->route($indexRouter)->withFlashSuccess(__('alerts.backend.users.created'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @return mixed
     */
    public function show(ManageUserRequest $request, User $user)
    {
        return view('backend.auth.user.show')
            ->withUser($user);
    }

    /**
     * @param ManageUserRequest    $request
     * @param RoleRepository       $roleRepository
     * @param PermissionRepository $permissionRepository
     * @param User                 $user
     *
     * @return mixed
     */
    public function edit(ManageUserRequest $request, RoleRepository $roleRepository, PermissionRepository $permissionRepository, User $user)
    {
        $userrole = $user->roles[0]->name; //dd($userrole);
        $userDetails = UserDetail::with(['user','parentDetails'])->where([['user_id',$user->id],['status','active']])->first();
        $sub_selectedUser = $selectedUser = $content ='';
        // if($userrole=='agent') {
        //     $superStockyuser = UserDetail::with(['user','parentDetails'])->where([['user_id',$userDetails->parent],['status','active']])->first();
        //     $selectedUser = $superStockyuser->parent; //dd($superStockyuser);
        //     $sub_selectedUser = $userDetails->parent; //dd($sub_selectedUser,$userDetails->user->id);
        //     $content = [$superStockyuser->user->id=>$superStockyuser->user->email];
        // } else {
            $selectedUser = isset($userDetails->parent)?$userDetails->parent:null;
       // }
        $parentRole = $this->userFunction->getParentRole($userrole);
        $paricularParentRoleId = ((count(auth()->user()->roles)>0)&&(auth()->user()->roles[0]->id!=1))?auth()->user()->id:0;
        
        return view('backend.auth.user.edit')
        ->withUsers($this->userRepository->getParentUser($paricularParentRoleId, 'id', 'asc',$parentRole))
            ->with(compact('userrole','userDetails','selectedUser','sub_selectedUser','content'))
            ->withUser($user)
            ->withRoles($roleRepository->where('id',$user->roles[0]->id)->get())
            ->withUserRoles($user->roles->pluck('name')->all())
            ->withPermissions($permissionRepository->get(['id', 'name']))
            ->withUserPermissions($user->permissions->pluck('name')->all());
    }

    /**
     * @param UpdateUserRequest $request
     * @param User              $user
     *
     * @throws \App\Exceptions\GeneralException
     * @throws \Throwable
     * @return mixed
     */
    public function update(UpdateUserRequest $request, User $user)
    {
        $roleType = $request->input('roles')[0];
        $this->userRepository->update($user, $request->only(
            'first_name',
            'last_name',
            'email',
            'roles',
            'permissions',
            'revenue_commission',
            'adminParent',
            'subchild'
        ));
        $indexRouter = $this->userFunction->getIndexRouter($roleType);
        return redirect()->route($indexRouter)->withFlashSuccess(__('alerts.backend.users.updated'));
    }

    /**
     * @param ManageUserRequest $request
     * @param User              $user
     *
     * @throws \Exception
     * @return mixed
     */
    public function destroy(ManageUserRequest $request, User $user)
    {
        $this->userRepository->deleteById($user->id);

        event(new UserDeleted($user));

        return redirect()->route('admin.auth.user.deleted')->withFlashSuccess(__('alerts.backend.users.deleted'));
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return mixed
     */
    public function adminindex(ManageUserRequest $request, $id = null)
    {
        $userIds = '';
        if($id != '') {
            $userIds = UserDetail::with(['user'])->where('parent',$id)->pluck('user_id');
            }
        $roleType = 'superAdmin';
        return view('backend.auth.user.index')
            ->with(compact('roleType'))
            ->withUsers($this->userRepository->getActivePaginated(25, 'id', 'asc','superAdmin',$userIds));
    }

    /**
     * @param ManageUserRequest $request
     *
     * @throws \Exception
     * @return mixed
     */
    public function userwithparticularparent(ManageUserRequest $request)
    {
        $param = $request->only('parentId');
        $sub_selectedUser = '';
        $content = $this->userRepository->getUserParticularParent($param);
        $view = view("backend.includes.dropbox",compact('content','sub_selectedUser'))->render();
        return response()->json(['html'=>$view]);

    }

    /**
     * @param ManageUserRequest $request
     *
     * @return mixed
     */
    public function superstockezindex(ManageUserRequest $request, $id = null)
    {
        
        $userIds = '';
        if($id != '') {
            $userIds = UserDetail::with(['user'])->where('parent',$id)->pluck('user_id');
            }
      
        $roleType = 'admin';
        return view('backend.auth.user.index')
            ->with(compact('roleType'))
            ->withUsers($this->userRepository->getActivePaginated(25, 'id', 'asc','admin',$userIds));
    }

    /**
     * @param ManageUserRequest $request
     *
     * @return mixed
     */
    public function stockezindex(ManageUserRequest $request, $id = null)
    {
        $userIds = null;
        //dd(auth()->user()->roles);

        if(((count(auth()->user()->roles)>0)&&(auth()->user()->roles[0]->id==2))&&($id=='')) {
        $userIds = UserDetail::with(['user'])->where('parent',auth()->user()->id)->pluck('user_id');
        }

        elseif(((count(auth()->user()->roles)>0)&&(auth()->user()->roles[0]->id==3))&&($id=='')) {
            $userIds = UserDetail::with(['user'])->where('user_id',auth()->user()->id)->pluck('user_id');
            }

        elseif($id != '') {
            $userIds = UserDetail::with(['user'])->where('parent',$id)->pluck('user_id');
        }
        $roleType = 'agent';
        return view('backend.auth.user.index')
            ->with(compact('roleType'))
            ->withUsers($this->userRepository->getActivePaginated(25, 'id', 'asc','agent',$userIds));
    }

  
}
